# Function to get instance metadata using IMDSv1
function Get-InstanceMetadata {
    param (
        [string]$path
    )
    try {
        $url = "http://169.254.169.254/latest/meta-data/$path"
        $response = Invoke-RestMethod -Uri $url -Method Get -TimeoutSec 5
        return $response
    } catch {
        # Do not output any error, just return $null
        return $null
    }
}

# Function to get instance metadata using IMDSv2
function Get-InstanceMetadataV2 {
    param (
        [string]$path
    )
    try {
        $tokenUrl = "http://169.254.169.254/latest/api/token"
        $token = Invoke-RestMethod -Uri $tokenUrl -Method Put -Headers @{"X-aws-ec2-metadata-token-ttl-seconds"="21600"}
        
        $url = "http://169.254.169.254/latest/meta-data/$path"
        $response = Invoke-RestMethod -Uri $url -Method Get -Headers @{"X-aws-ec2-metadata-token"=$token} -TimeoutSec 5
        return $response
    } catch {
        Write-Error "Failed to get instance metadata using IMDSv2: $_"
        return $null
    }
}

# Start timer
$startTime = Get-Date

# Function to check if the script has timed out
function Check-Timeout {
    param (
        [datetime]$startTime,
        [int]$timeoutSeconds
    )
    $currentTime = Get-Date
    $elapsedTime = ($currentTime - $startTime).TotalSeconds
    if ($elapsedTime -ge $timeoutSeconds) {
        Write-Error "Script execution timed out after $timeoutSeconds seconds."
        exit 1
    }
}

# Attempt to get instance ID using IMDSv1
$instanceId = Get-InstanceMetadata -path "instance-id"

# Check for timeout
Check-Timeout -startTime $startTime -timeoutSeconds 30

if (-not $instanceId) {
    # If IMDSv1 fails, try IMDSv2
    $instanceId = Get-InstanceMetadataV2 -path "instance-id"
    
    # Check for timeout
    Check-Timeout -startTime $startTime -timeoutSeconds 30
}

if ($instanceId) {
    try {
        # Get tags using AWS CLI
        $tags = aws ec2 describe-tags --filters "Name=resource-id,Values=$instanceId"
        
        # Check for timeout
        Check-Timeout -startTime $startTime -timeoutSeconds 30
        
        #Write-Output "Tags for instance $instanceId:"
        Write-Output $tags
    } catch {
        Write-Error "Failed to get tags: $_"
    }
} else {
    Write-Error "Failed to get instance ID using both IMDSv1 and IMDSv2."
}