Param(
  [Parameter(Mandatory = $true)]
  [string]$Hostname,

  [Parameter(Mandatory = $true)]
  [string]$AdminUserName,

  [Parameter(Mandatory = $true)]
  [string]$AdminPassword,

  [Parameter(Mandatory = $false)]
  [int]$AdminWebPort = 8443,

  [Parameter(Mandatory = $false)]
  [int]$SoapPort = 10001
)

$ErrorActionPreference = 'Stop'

function New-CerberusProxy {
  param(
    [Parameter(Mandatory)][string]$Hostname,
    [Parameter(Mandatory)][string]$AdminUserName,
    [Parameter(Mandatory)][string]$AdminPassword,
    [Parameter(Mandatory)][int]$AdminWebPort,
    [Parameter(Mandatory)][int]$SoapPort
  )

  [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12

  if (-not ("Dummy" -as [type])) {
    add-type -TypeDefinition @"
using System;
using System.Net;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
public static class Dummy {
  public static bool ReturnTrue(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors) { return true; }
  public static RemoteCertificateValidationCallback GetDelegate() {
    return new RemoteCertificateValidationCallback(Dummy.ReturnTrue);
  }
}
"@
  }
  [System.Net.ServicePointManager]::ServerCertificateValidationCallback = [Dummy]::GetDelegate()

  $wsdlUrl    = "https://$Hostname`:$AdminWebPort/wsdl/Cerberus.wsdl"
  $serviceUrl = "https://$Hostname`:$SoapPort/service/cerberusftpservice"

  $secure   = ConvertTo-SecureString $AdminPassword -AsPlainText -Force
  $httpCred = New-Object System.Management.Automation.PSCredential($AdminUserName, $secure)

  $svc = New-WebServiceProxy -Uri $wsdlUrl -Class CerberusFtp -Namespace CerberusFtp -Credential $httpCred
  if ($null -eq $svc) { throw "New-WebServiceProxy returned null." }

  $svc.Url = $serviceUrl
  $svc.PreAuthenticate = $true

  $netCred = New-Object System.Net.NetworkCredential($AdminUserName, $AdminPassword)
  $cache   = New-Object System.Net.CredentialCache
  $uri     = [Uri]$serviceUrl
  $cache.Add($uri, "NTLM",     $netCred)
  $cache.Add($uri, "Kerberos", $netCred)
  $cache.Add($uri, "Basic",    $netCred)
  $svc.Credentials = $cache

  return $svc
}

try {
  $svc = New-CerberusProxy -Hostname $Hostname -AdminUserName $AdminUserName -AdminPassword $AdminPassword -AdminWebPort $AdminWebPort -SoapPort $SoapPort

  [CerberusFtp.GetLicenseInfoRequest] $req = New-Object CerberusFtp.GetLicenseInfoRequest
  $req.credentials = New-Object CerberusFtp.Credentials
  $req.credentials.user     = $AdminUserName
  $req.credentials.password = $AdminPassword

  [CerberusFtp.GetLicenseInfoResponse] $resp = $svc.GetLicenseInfo($req)
  if (-not $resp) { throw "Empty response from GetLicenseInfo" }

  $li = $resp.LicenseInformation
  if (-not $li) { throw "LicenseInformation not present in response" }

  $issuedDate = $null
  if ($li.issuedDate) {
    $issuedDate = [DateTimeOffset]::FromUnixTimeSeconds([int64]$li.issuedDate).DateTime
  } else {
    throw "issuedDate not present in LicenseInformation"
  }

  $daysValid = [int]$li.daysValid
  if ($daysValid -le 0) { throw "daysValid is not a positive integer" }

  $expiration = $issuedDate.AddDays($daysValid)
  $daysRemain = [math]::Floor(($expiration - (Get-Date)).TotalDays)

  Write-Output $daysRemain
}
catch {
  Write-Output -1
  exit 1
}
